<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\Controller;
use App\Libraries\Tabler;
use App\Models\Branch;
use App\Models\Role;
use App\Models\User;
use App\Rules\AlphaSpace;
use App\Rules\ArabicAlphaSpace;
use App\Rules\Mobile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx\Rels;
use WebReinvent\CPanel\CPanel;

class UsersController extends Controller
{
    /**
     * List users
     *
     * @param Request $request
     * @return void
     */
    public function index(Request $request)
    {
        $table = $this->getUsersTable($request);

        $data = $table->initTable();

        $this->loadView('users.index', 'users.title', 'users', [], $data);
    }
    //---------------------------------------------------------------------------------
    /**
     * Add new user
     *
     * @param Request $request
     * @return void
     */
    public function add(Request $request)
    {
        if ($request->getMethod() == 'POST') {

            $rules = [
                'first-name' => ['required', 'min:3', 'max:50', new ArabicAlphaSpace],
                'last-name' => ['required', 'min:3', 'max:50', new ArabicAlphaSpace],
                'email' => ['required', 'email', 'unique:users,email'],
                'password' => ['required', 'min:6', 'max:20'],
                'cpassword' => ['required', 'same:password'],
                'mobile' => ['required', 'unique:users,mobile', new Mobile],
                'role' => ['required', 'integer', 'exists:roles,id'],
                'branches' => ['required'],
            ];

            if (!is_admin()) {
                $rules['branches.*'] = ['integer', 'exists:branches,id', Rule::in(allowed_branches())];
            } else {
                $rules['branches.*'] = ['integer', 'exists:branches,id'];
            }


            $validated = $request->validate($rules);


            $user_info['first_name'] = $validated['first-name'];
            $user_info['last_name'] = $validated['last-name'];
            $user_info['mobile'] = $validated['mobile'];
            $user_info['email'] = $validated['email'];
            $user_info['password'] = Hash::make($validated['password']);
            $user_info['role_id'] = $validated['role'];

            if (array_key_exists('blocked', $request->all())) {

                $user_info['is_blocked'] = true;
            } else {

                $user_info['is_blocked'] = false;
                $user_info['login_attempts'] = 0;
            }

            if (array_key_exists('notify', $request->all())) {

                // Send login info to user


            }

            DB::transaction(function () use ($user_info, $validated) {

                $user = User::create($user_info);

                $user->branches()->attach($validated['branches']);
            });

            $email = $validated['email'];
            $arr = explode('@', $email);
            $email_domain = array_pop($arr);
            $email_address = array_pop($arr);

            if ($email_domain ==  'zain-clinic.com') {

                $cpanel = new CPanel();
                $module = 'Email';
                $function = 'add_pop';
                $parameters = [
                    'email' => $email_address,
                    'password' => $validated['password'], //make sure you use strong password
                    'quota' => '500',
                ];

                $response = $cpanel->callUAPI($module, $function, $parameters);
            }

            return redirect()->route('admin.users')->with('success', __('admin/common.msgs.success.add'));
        }

        // Get roles & services
        $data['roles'] = Role::select('id', 'role')->orderBy('role')->get();
        $data['branches'] = Branch::select('id', 'branch')->orderBy('branch')->get();
        //$data['services'] = Service::orderBy('service')->get();

        // Generate random password
        $data['gen_password'] = Str::random(10);

        $this->loadView('users.add', 'users.add_title', 'users.add', [], $data, ['bs-select/bootstrap-select.min.css'], ['bs-select/bootstrap-select.min.js']);
    }
    //---------------------------------------------------------------------------------

    /**
     * Edit existing user
     *
     * @param Request $request
     * @param User $user
     * @return void
     */
    public function edit(Request $request, User $user)
    {
        if (!is_admin() && !in_array($user->id, allowed_users())) {
            abort(404);
        }

        if ($request->getMethod() == 'POST') {

            $rules = [
                'first-name' => ['required', 'min:3', 'max:50', new ArabicAlphaSpace],
                'last-name' => ['required', 'min:3', 'max:50', new ArabicAlphaSpace],
                'email' => ['required', 'email', Rule::unique('users', 'email')->ignore($user->id)],
                'password' => ['nullable', 'string', 'min:6', 'max:20'],
                'cpassword' => ['same:password'],
                'mobile' => ['required', Rule::unique('users', 'mobile')->ignore($user->id), new Mobile],
                'role' => ['required', 'integer', 'exists:roles,id'],
                //'services' => ['required_with:is-physician'],
                //'services.*' => ['integer', 'exists:services,id'],
                'branches' => ['required'],
                'branches.*' => ['integer', 'exists:branches,id'],
            ];

            $validated = $request->validate($rules);

            $user_info['first_name'] = $validated['first-name'];
            $user_info['last_name'] = $validated['last-name'];
            $user_info['mobile'] = $validated['mobile'];
            $user_info['email'] = $validated['email'];
            $user_info['role_id'] = $validated['role'];

            if (!empty(trim($validated['password']))) {

                $user_info['password'] = Hash::make($validated['password']);

                $email = $validated['email'];
                $arr = explode('@', $email);
                $email_domain = array_pop($arr);

                if ($email_domain ==  'zain-clinic.com') {

                    $cpanel = new CPanel();
                    $module = 'Email';
                    $function = 'passwd_pop';
                    $parameters = [
                        'email' => $validated['email'],
                        'password' => $validated['password'], //make sure you use strong password

                    ];

                    $response = $cpanel->callUAPI($module, $function, $parameters);
                }
            }

            if (array_key_exists('blocked', $request->all())) {

                $user_info['is_blocked'] = true;

                // block email login on cpanel
                $cpanel = new CPanel();
                $module = 'Email';
                $function = 'suspend_login';
                $parameters = [
                    'email' => $validated['email'],
                ];

                $response = $cpanel->callUAPI($module, $function, $parameters);
            } else {

                $user_info['is_blocked'] = false;
                $user_info['login_attempts'] = 0; // reset login attempts count

                // un-block email login on cpanel
                $cpanel = new CPanel();
                $module = 'Email';
                $function = 'unsuspend_login';
                $parameters = [
                    'email' => $validated['email'],
                ];

                $response = $cpanel->callUAPI($module, $function, $parameters);
            }

            if (array_key_exists('notify', $request->all())) {

                // Send login info to user

            }

            // Branches
            DB::transaction(function () use ($user, $user_info, $validated) {

                User::where('id', $user->id)->update($user_info);

                $user->branches()->sync($validated['branches']);
            });

            return redirect()->route('admin.users')->with('success', __('admin/common.msgs.success.edit'));
        }


        $data['roles'] = Role::select('id', 'role')->orderBy('role')->get();

        if (!is_admin()) {
            $data['branches'] = Branch::select('id', 'branch')->whereIn('id', allowed_branches())->orderBy('branch')->get();
        } else {
            $data['branches'] = Branch::select('id', 'branch')->orderBy('branch')->get();
        }


        // Generate random password
        $data['gen_password'] = Str::random(10);

        $data['user_info'] = User::where('is_admin', 0)->where('id', $user->id)->first();

        //$services = $user->services;
        $branches = $user->branches;

        //$selected_services = [];
        $selected_branches = [];

        if ($branches->count()) {
            foreach ($branches as $branch) {
                $selected_branches[] = $branch->id;
            }
        }

        //$data['selected_services'] = json_encode($selected_services,JSON_UNESCAPED_UNICODE);
        //$data['selected_services'] = implode("','", $selected_services);
        $data['selected_branches'] = implode("','", $selected_branches);

        $this->loadView('users.edit', 'users.edit_title', 'users.edit', $user, $data, ['bs-select/bootstrap-select.min.css'], ['bs-select/bootstrap-select.min.js']);
    }
    //-------------------------------------------------------------
    public function delete(Request $request, User $user)
    {

        if ($user->complaintComments && $user->complaintComments->count() > 0) {

            return redirect()->route('admin.users')->with('danger', __('admin/common.msgs.error.delete'));
        }

        if ($user->patientComments && $user->patientComments->count() > 0) {

            return redirect()->route('admin.users')->with('danger', __('admin/common.msgs.error.delete'));
        }

        $user->delete();

        return redirect()->route('admin.users')->with('success', __('admin/common.msgs.success.delete'));
    }
    /*======================================================*/
    private function getUsersTable(Request $request)
    {

        $route_key = 'admin.users';
        $model = new User();
        $model = $model->join('roles', 'users.role_id', '=', 'roles.id');
        $model = $model->where('is_admin', 0);

        if (!is_admin()) {
            $model = $model->whereIn('users.id', allowed_users());
        }

        $yes = __('admin/common.text.yes');
        $no = __('admin/common.text.no');

        $blocked = DB::raw("IF(is_blocked=true,'" . $yes . "', '" . $no . "') as is_blocked");

        $th = ['first_name', 'last_name', 'email', 'role','is_blocked'];
        $select = ['users.id', 'first_name', 'last_name', 'email','mobile','roles.role as role', $blocked];
        $sortable = ['first_name', 'last_name', 'email', 'role','is_blocked'];
        $searchable = ['first_name', 'last_name', 'email', 'role'];

        $table = new Tabler($route_key, $model, $select, $th, $sortable, $searchable, $request);

        return $table;
    }

    /*================================================*/
    public function print(Request $request, $selection)
    {
        if ($selection == 'all') {

            $yes = __('admin/common.text.yes');
            $no = __('admin/common.text.no');

            $blocked = DB::raw("IF(is_blocked=true,'" . $yes . "', '" . $no . "') as is_blocked");

            $select = ['users.id', 'first_name', 'last_name', 'email','mobile','roles.role as role', $blocked];

            $obj = User::allowedUsers()->select($select)
                ->join('roles', 'users.role_id', '=', 'roles.id')
                ->where('is_admin', 0)
                ->orderBy('first_name')
                ->orderBy('last_name');

            $th = ['first_name', 'last_name', 'email', 'role','is_blocked'];


            // to remove th stdClass
            $data['rows'] = (array) json_decode(json_encode($obj->get()->toArray()), true);

            foreach ($th as $db_col) {
                $data['columns'][$db_col] = '<th>' . __('admin/users.columns.' . $db_col) . '</th>';
            }
        } else {

            $table = $this->getUsersTable($request);

            $result = $table->initTable();

            $rows = $result['result']->toArray();

            $data['rows'] = $rows['data'];

            $data['columns'] = $result['columns'];
        }

        $data['_page_title'] = __('admin/users.title');

        echo view('admin.printables.list', $data);
    }
}
