<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\Controller;
use App\Libraries\Tabler;
use App\Models\Appointment;
use App\Models\Complaint;
use App\Models\Patient;
use App\Models\PatientAttachments;
use App\Models\PatientComment;
use App\Models\PatientPretest;
use App\Models\User;
use App\Rules\GeneralText;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;


class PatientsInfoController extends Controller
{
    //==========================================================
    // Info Tabs
    //==========================================================
    /**
     * Get patient info
     *
     * @param Request $request
     * @param Patient $patient
     * @return void
     */
    public function index(Request $request, Patient $patient)
    {
        if ( !is_admin() && !in_array($patient->id,allowed_patients()) ){
            abort(404);
        }

        $this->comments($request, $patient);
    }
    //------------------------------------------------------------
    /**
     * Patient Comments pages
     *
     * @param Request $request
     * @param Patient $patient
     * @return void
     */
    public function comments(Request $request, Patient $patient)
    {
        if ($request->method() == 'POST') {

            $rules = [
                'patient-id' => ['required', 'exists:patients,id'],
                'comment' => ['required', 'min:5', 'max:4000', new GeneralText],
            ];

            $validated = $request->validate($rules);

            $data['comment'] = $validated['comment'];
            $data['added_by'] = user_id();
            $data['patient_id'] = $patient->id;

            $data['comment'] = PatientComment::create($data);

            return redirect()->route('admin.patients.info.comments', $patient);
        }

        $data['patient'] = $patient;
        $data['comments'] = $patient->comments;

        $data['patient_history'] = [];

        $data['tab_content_page'] = 'comments';

        $this->loadView('patients.info', 'patients.info_title', 'patients.info', $patient, $data);
    }

    //------------------------------------------------------------
    public function appointments(Request $request, Patient $patient)
    {
        
        $model = new Appointment();

        $select = ['ref_number', 'appointments.id as id', 'appointment_date', 'start_time', 'treatments.treatment AS treatment', 'branches.branch as branch'];

        $model = $model->allowedPatients();
        
        $model = $model->join('treatments', 'appointments.treatment_id', '=', 'treatments.id');

        $model = $model->join('branches', 'appointments.branch_id', '=', 'branches.id');

        $model = $model->join('patients', 'appointments.patient_id', '=', 'patients.id');

        $model = $model->where('appointments.postponed', '=', 0);

        $model = $model->where('appointments.patient_id', '=', $patient->id);

        $model = $model->orderBy('appointment_date')->orderBy('start_time');

        $th = ['ref_number', 'treatment', 'branch', 'appointment_date', 'start_time'];

        $sortable = ['ref_number', 'appointment_date', 'treatment', 'branch', 'start_time'];

        $searchable = ['ref_number', 'appointment_date', 'treatment', 'branch'];

        $table = new Tabler('admin.appointments', $model, $select, $th, $sortable, $searchable, $request, true);

        $data = $table->initTable();

        $data['patient'] = $patient;
        $data['appointments'] = $patient->appointments;
        $data['tab_content_page'] = 'appointments';

        $this->loadView('patients.info', 'patients.info_title', 'patients.info', $patient, $data);
    }
    //------------------------------------------------------------
    /*=====================
    *  Patient Pretests
    =======================*/
    public function pretests(Request $request, Patient $patient)
    {
        $model = new PatientPretest();

        $user = new User();

        $user_full_name = $user->selectFullName();

        $select_columns = ['patient_pretest.id', DB::raw('DATE_FORMAT(patient_pretest.created_at, "%Y-%m-%d") as form_date'), 'patient_pretest.title', $user_full_name];

        $model = $model->join('patients', 'patient_pretest.patient_id', '=', 'patients.id');
        $model = $model->join('pretests', 'patient_pretest.pretest_id', '=', 'pretests.id');
        $model = $model->join('users', 'patient_pretest.created_by', '=', 'users.id');

        $model = $model->where('patient_id',$patient->id);
        
        $th_columns = ['form_date', 'title', 'user_full_name'];

        $sortable_columns = ['title', 'form_date', 'user_full_name'];
        $searchable_columns = ['title', 'form_date', 'user_full_name'];

        $route_key = 'admin.pretests';

        $table = new Tabler($route_key, $model, $select_columns, $th_columns, $sortable_columns, $searchable_columns, $request);

        $data = $table->initTable();

        $data['patient'] = $patient;
        
        $data['tab_content_page'] = 'pretests';

        $this->loadView('patients.info', 'patients.info_title', 'patients.info', $patient, $data);
    }
    //------------------------------------------------------------
    /*=====================
    *  Patient Complaints
    =======================*/
    //------------------------------------------------------------
    public function complaints(Request $request, Patient $patient)
    {
        $model = new Complaint();

        $yes = __('admin/common.text.yes');
        $no = __('admin/common.text.no');
        
        $is_resolved = DB::raw("IF(is_resolved=0,'".$no."', '".$yes."') AS is_resolved");

        $select =  ['complaints.id','complaint_date','description','patient_id',$is_resolved,'complaints.created_at','complaints.updated_at'];

        $model = $model->join('patients', 'complaints.patient_id', '=', 'patients.id');

        $model = $model->where('complaints.patient_id', $patient->id);

        $th = ['complaint_date','description', 'is_resolved', 'created_at', 'updated_at'];

        $sortable = ['complaint_date'];
        
        $searchable = ['description','is_resolved'];

        $table = new Tabler('admin.complaints', $model, $select, $th, $sortable, $searchable, $request,true);

        $data = $table->initTable();

        $data['patient'] = $patient;
        $data['tab_content_page'] = 'complaints';

        $this->loadView('patients.info', 'patients.info_title', 'patients.info', $patient, $data);
    }

    //------------------------------------------------------------
    /*=====================
    *  Patient Attachments
    =======================*/
    public function attachments(Request $request, Patient $patient)
    {
        $data['patient'] = $patient;
        //$data['pretests'] = Pretest::orderBy('title')->get();
        $data['attachments'] = $patient->attachments;
        $data['tab_content_page'] = 'attachments';

        $this->loadView('patients.info', 'patients.info_title', 'patients.info', $patient, $data, ['bs-datepicker/dp-bs-4.min.css'], ['momentjs/moment.min.js', 'bs-datepicker/dp-bs-4.min.js']);
    }
    //------------------------------------------------------------
    public function addAttachment(Request $request, Patient $patient)
    {
        if ($request->getMethod() == 'POST') {

            $rules = [
                'attachment-title' => ['required', 'max:50', new GeneralText],
                'patient-attachment' => ['required','mimes:png,jpg,jpeg,pdf,doc,docx', 'min:10', 'max:3096']
            ];

            $validated = $request->validate($rules);

            $attachment = [
                'patient_id' => $patient->id,
                'title' => $validated['attachment-title'],
            ];

            if ($request->file('patient-attachment') && $request->file('patient-attachment')->isValid()) {

                $patient_dir = config('admin.patient_attachments_dir') . '/' . $patient->id;

                $file_name = md5(time() . $request->file('patient-attachment')->guessExtension()) . '.' . $request->file('patient-attachment')->guessExtension();

                $request->file('patient-attachment')->storeAs($patient_dir, $file_name);

                $attachment['path'] = $file_name;
            }

            PatientAttachments::create($attachment);

            return redirect()->route('admin.patients.info.attachments', $patient->id)->with('success', __('admin/common.msgs.success.add'));
        }

        $data['patient'] = $patient;
        $data['attachments'] = $patient->attachments;
        $data['tab_content_page'] = 'attachments';

        $this->loadView('patients.info', 'patients.info_title', 'patients.info', $patient, $data, ['bs-datepicker/dp-bs-4.min.css'], ['momentjs/moment.min.js', 'bs-datepicker/dp-bs-4.min.js']);
    }

    //------------------------------------------------------------
    public function downloadAttachment(Request $request, Patient $patient, $attachment_id)
    {

        $attachment = PatientAttachments::findOrFail($attachment_id);

        // if attachment is for the same selected user
        if ($attachment->patient_id != $patient->id) {
            abort(403);
        }

        $path = config('admin.patient_attachments_dir') . '/' . $attachment->patient_id . '/' . $attachment->path;

        if (Storage::exists($path)) {

            return Storage::download($path);
        }

        abort(404);
    }
    //------------------------------------------------------------
    public function deleteAttachment(Request $request, Patient $patient, $attachment_id)
    {

        $attachment = PatientAttachments::findOrFail($attachment_id);

        // if attachment is for the same selected user
        if ($attachment->patient_id != $patient->id) {
            abort(403);
        }

        $path = config('admin.patient_attachments_dir') . '/' . $attachment->patient_id . '/' . $attachment->path;

        if (Storage::exists($path)) {

            $attachment->delete();

            Storage::delete($path);

            return redirect()->route('admin.patients.info.attachments', $patient->id)->with('success', __('admin/common.msgs.success.delete'));
        }

        abort(404);
    }
    //------------------------------------------------------------
    public function sms(Request $request, Patient $patient)
    {
    }

    //------------------------------------------------------------
    public function notifications(Request $request, Patient $patient)
    {

    }

    //------------------------------------------------------------
}
