<?php

namespace App\Http\Controllers\Admin;

use App\Exports\ExportAppointments;
use App\Http\Controllers\Admin\Controller;
use App\Http\Controllers\Api\Appointments;
use App\Libraries\AppointmentsCommon;
use App\Libraries\Tabler;
use App\Models\Appointment;
use App\Models\Branch;
use App\Models\Patient;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;

/**
 * Appointment Rules:
 *
 *  - Patients are not allowed to book an appointment on the same day
 *  - Patients can't book multiple appointments on the same date in different branches
 *  - Patients CAN book multiple appointments for different treatments IN THE SAME BRANCH
 *  - Patients can't book the same treatment on the same day
 *
 */
class AppointmentsController extends Controller
{

    public function index(Request $request)
    {

        $table = $this->getAppointmentsTable($request);

        $data = $table->initTable();

        $this->loadView('appointments.index', 'appointments.title', 'appointments', [], $data);
    }
    /*============================================================== */
    public function add(Request $request)
    {

        if ($request->getMethod() == 'POST' && $request->ajax()) {

            $app_common = new AppointmentsCommon();

            $app_common->addAppointment($request);

            return $app_common->getResponse();
        }

        $data['patients'] = Patient::allowedPatients()->get();

        $this->loadView('appointments.add', 'appointments.add_title', 'appointments.add', [], $data, ['bs-select/bootstrap-select.min.css'], ['bs-select/bootstrap-select.min.js']);
    }
    /*============================================================== */
    public function postpone(Request $request, Appointment $appointment)
    {
        // If changing past appointment
        $app_date_before_update = $appointment->appointment_date;
        $app_start_time_before_update = date('H:i', strtotime($appointment->start_time));
        $today = date('Y-m-d');
        $time = date('H:i');

        if ($app_date_before_update < $today || ($app_date_before_update == $today && $app_start_time_before_update < $time)) {

            return redirect()->back()->with('danger', __('admin/appointments.error.passed_app'));
            //return redirect()->route('admin.appointments')->with('danger', __('admin/appointments.error.passed_app'));
        }

        if ($request->getMethod() == 'POST' && $request->ajax() && in_array($appointment->branch_id, allowed_branches())) {

            $app_common = new AppointmentsCommon();

            $app_common->changeAppointment($request, $appointment);

            return $app_common->getResponse();
        }

        /* if (!in_array($appointment->branch_id, allowed_branches())) {

            abort('404');
        } */

        $model = new Appointment();

        $data['appointment'] = $model->getAppointmentByID($appointment->id);

        $data['branches'] = Branch::orderBy('branch')->whereIn('id', allowed_branches())->get();

        $data['treatments'] = Branch::with(['treatments' => function ($q) {
            $q->orderBy('treatment');
        }])->find($appointment->branch_id)->treatments;

        $app_common = new AppointmentsCommon();

        $calendar = $app_common->generateCalendar($appointment->patient_id, $appointment->branch_id, $appointment->treatment_id, 0);

        $calendar['selected_date'] = $appointment->appointment_date;
        $calendar['selected_time'] = date('h:i A', strtotime($appointment->start_time));

        $data['app_table'] = view('admin.appointments.table', $calendar);

        $this->loadView('appointments.postpone', 'appointments.postpone_title', 'appointments.postpone', $appointment, $data, ['bs-select/bootstrap-select.min.css'], ['bs-select/bootstrap-select.min.js']);
    }
    /*============================================================== */
    /**
     * Single appointment info
     *
     * @param Request $request
     * @param Appointment $appointment
     * @return void
     */
    public function info(Request $request, Appointment $appointment)
    {
        if ($request->ajax()) {

            $model = new Appointment;

            $data['appointment'] = $model->getAppointmentByID($appointment->id);

            return response()->view('admin.appointments.info_modal', $data);
        }

        return redirect()->route('admin.appointments');
    }

    /*============================================================== */
    public function delete(Request $request, $appointment_id)
    {

        $appointment = Appointment::findOrFail($appointment_id);

        $app_date_before_update = $appointment->appointment_date;
        $app_start_time_before_update = date('H:i', strtotime($appointment->start_time));
        $today = date('Y-m-d');
        $time = date('H:i');

        if ($app_date_before_update < $today || ($app_date_before_update == $today && $app_start_time_before_update < $time)) {

            return redirect()->back()->with('danger', __('admin/appointments.error.passed_app'));
            //return redirect()->route('admin.appointments')->with('danger', __('admin/appointments.error.passed_app'));
        }

        if ($appointment->checked_in_at || $appointment->admitted_at) {
            return redirect()->back()->with('danger', __('admin/appointments.error.passed_app'));
            //return redirect()->route('admin.appointments')->with('danger', __('admin/appointments.error.passed_app'));
        }

        $review = $appointment->review;

        if ($review && $review->count() > 0) {
            return redirect()->back()->with('danger', __('admin/appointments.error.passed_app'));
            //return redirect()->route('admin.patients')->with('danger', __('admin/common.msgs.error.delete'));
        }

        $appointment->delete();

        return redirect()->back()->with('success', __('admin/common.msgs.success.delete'));
        //return redirect()->route('admin.appointments')->with('success', __('admin/common.msgs.success.delete'));
    }
    /*============================================================== */
    public function print(Request $request, $selection)
    {

        if ($selection == 'all') {

            $model = new Appointment();

            $select = $model->appointmentsListColumns();

            $obj = Appointment::allowedBranches()->select($select)
                                ->join('treatments', 'appointments.treatment_id', '=', 'treatments.id')
                                ->join('branches', 'appointments.branch_id', '=', 'branches.id')
                                ->join('patients', 'appointments.patient_id', '=', 'patients.id')
                                ->orderBy('appointment_date','DESC')
                                ->where('appointments.postponed', '=', 0);

            $th = ['appointment_date', 'patient_name', 'patient_mobile', 'treatment', 'start_time','is_scheduled'];

            if ( is_admin() ||  count(allowed_branches()) > 1) {

                $th[] = 'branch';

            }

            // to remove th stdClass
            $data['rows'] = (array) json_decode(json_encode($obj->get()->toArray()), true);

            foreach ($th as $db_col) {
                $data['columns'][$db_col] = '<th>' . __('admin/appointments.columns.' . $db_col) . '</th>';
            }
        } else {

            $table = $this->getAppointmentsTable($request);

            $result = $table->initTable();

            $rows = $result['result']->toArray();

            $data['rows'] = $rows['data'];

            $data['columns'] = $result['columns'];
        }

        $data['_page_title'] = __('admin/reports.patient_history.title');

        echo view('admin.printables.list', $data);
    }
    /*============================================================== */
    private function getAppointmentsTable(Request $request){

        $model = new Appointment();

        $select = $model->appointmentsListColumns();

        if ($request->get('pid')) {

            $patient_id = (int) $request->get('pid');

            if (!in_array($patient_id, allowed_patients())) {
                abort('404');
            }

            $model = $model->where('appointments.patient_id', $patient_id);
        }

        $model = $model->allowedBranches();

        $model = $model->join('treatments', 'appointments.treatment_id', '=', 'treatments.id');

        $model = $model->join('branches', 'appointments.branch_id', '=', 'branches.id');

        $model = $model->join('patients', 'appointments.patient_id', '=', 'patients.id');

        $model = $model->where('appointments.postponed', '=', 0);

        // Filter by user branch permissions
        //$model = $model->whereIn('branch_id', allowed_branches());

        //$th = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'treatment', 'start_time','is_scheduled'];
        $th = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'treatment', 'start_time','notes'];

        //$sortable = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'treatment', 'start_time','is_scheduled'];
        $sortable = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'treatment', 'start_time','notes'];

        $searchable = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'notes','treatment'];

        if ( is_admin() ||  count(allowed_branches()) > 1) {

            $th[] = 'branch';
            $sortable[] ='branch';
            $searchable[] ='branch';

        }

        $table = new Tabler('admin.appointments', $model, $select, $th, $sortable, $searchable, $request, true);

        return $table;
    }

    /*============================================================== */
    /****************************
     *  Ajax Functions
     ****************************/
    public function getBranches(Request $request)
    {
        if ($request->ajax()) {

            $branches = Branch::allowedBranches()->orderBy('branch')->get();

            if ($branches->count() > 0) {
                return response()->json(['treatments' => $branches->toArray()]);
            } else {
                return response()->json(['error' => __('admin/common.msgs.no_results')], 404);
            }
        }
    }
    /*=================================================*/
    public function getTreatments(Request $request, int $branch_id = 1)
    {
        if ($request->ajax()) {

            if (!in_array($branch_id, allowed_branches())) {
                return response()->json(['error' => __('admin/common.msgs.no_results')], 404);
            }

            $treatments = Branch::with(['treatments' => function ($q) {
                $q->orderBy('treatment', 'asc');
            }])->find($branch_id)->treatments;

            if ($treatments->count() > 0) {
                return response()->json(['treatments' => $treatments->toArray()]);
            } else {
                return response()->json(['error' => __('admin/common.msgs.no_results')], 404);
            }
        }
    }

    /*=================================================*/
    public function getCalendar(Request $request, $patient_id, $branch_id, $treatment_id, $week_number)
    {
        if ($request->ajax()) {

            if (!in_array($branch_id, allowed_branches())) {
                return response()->json(['error' => __('admin/common.msgs.no_results')], 404);
            }

            // Validate route variables
            $rules = [
                'patient_id' => ['required', 'exists:patients,id'],
                'branch_id' => ['required', 'exists:branches,id', Rule::in(allowed_branches())],
                'treatment_id' => ['required', Rule::exists('branch_treatment', 'treatment_id')->where('branch_id', $branch_id)],
                'week_number' => ['required', 'min:0', 'max:3'],
            ];

            $validator = Validator::make($request->route()->parameters(), $rules);

            if ($validator->fails()) {

                return response()->json(['error' => $validator->errors()->all()], 400);
            }

            //------------------------------------------------
            $app_common = new AppointmentsCommon();
            $data = $app_common->generateCalendar($patient_id, $branch_id, $treatment_id, $week_number);

            return response()->view('admin.appointments.table', $data);
        }
    }
    /*=================================================*/

    public function export(){

        return Excel::download(new ExportAppointments(), 'appointments.xlsx');

    }

}
